/*
 * <N_OCO_COPYRIGHT>
 * Licensed Materials - Property of IBM
 * 
 * 5724-H72, 5655-R36, 5724-L26, 5655-L82     
 * 
 * (c) Copyright IBM Corp. 2008 All Rights Reserved.
 * 
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with
 * IBM Corp.
 * <NOC_COPYRIGHT>
 */

/**
 * "JmqiGet" is a sample JMQI program to get messages from a          
 * message queue, and is an example of MQGET.                     
 *                                                                  
 *      -- sample reads from message queue named in the parameter   
 *                                                                  
 *      -- displays the contents of the message queue,              
 *         assuming each message data to represent a line of        
 *         text to be written                                       
 *                                                                  
 *         messages are removed from the queue                      
 *                                                                  
 *      -- writes a message for each MQI reason other than          
 *         MQRC_NONE; stops if there is a MQI completion code       
 *         of MQCC_FAILED                                           
 *
 *
 *   Program logic:                                                 
 *      Take name of input queue from the parameter                 
 *      MQOPEN queue for INPUT                                      
 *      while no MQI failures,
 *      .  MQGET next message, remove from queue                    
 *      .  print the result                                         
 *      .  (no message available counts as failure, and loop ends)  
 *      MQCLOSE the subject queue  
 *      
 *
 *
 * "JmqiGet" has the following parameters                          
 *       required:                                                  
 *                 (1) The name of the source queue                 
 *       optional:                                                  
 *                 (2) Queue manager name                           
 *                 (3) The open options                             
 *                 (4) The close options                            
 *                                                                                                         
 *
 * To connect to the Queue manager in Client mode, specify "MQSERVER" as an environment variable 
 * or a system parameter as follows:
 *     MQSERVER=ChannelName/TransportType/ConnectionName                                                                                                         
 *                                                                                                         
 *                                                      
 *                                                                                                         
 * For example:
 *     bindings mode:
 *         java com.ibm.mq.jmqi.samples.JmqiGet QUEUE QMGR
 *         
 *     client mode:
 *         java -DMQSERVER=SYSTEM.DEF.SVRCONN/TCP/localhost(1414) com.ibm.mq.jmqi.samples.JmqiGet QUEUE QMGR          
 */
package com.ibm.mq.jmqi.samples;

import java.nio.ByteBuffer;

import com.ibm.mq.constants.CMQC;
import com.ibm.mq.jmqi.JmqiEnvironment;
import com.ibm.mq.jmqi.JmqiFactory;
import com.ibm.mq.jmqi.JmqiMQ;
import com.ibm.mq.jmqi.JmqiPropertyHandler;
import com.ibm.mq.jmqi.JmqiThreadPoolFactory;
import com.ibm.mq.jmqi.MQCNO;
import com.ibm.mq.jmqi.MQGMO;
import com.ibm.mq.jmqi.MQMD;
import com.ibm.mq.jmqi.MQOD;
import com.ibm.mq.jmqi.handles.Hconn;
import com.ibm.mq.jmqi.handles.Hobj;
import com.ibm.mq.jmqi.handles.Phconn;
import com.ibm.mq.jmqi.handles.Phobj;
import com.ibm.mq.jmqi.handles.Pint;

/**
 * Sample program to put a message to a queue
 */
public class JmqiGet extends SampleFramework {

  // @COPYRIGHT_START@
  /** Comment for copyright_notice */
  static final String copyright_notice = "Licensed Materials - Property of IBM "
      + "5724-H72, 5655-R36, 5724-L26, 5655-L82                "
      + "(c) Copyright IBM Corp. 2008 All Rights Reserved. "
      + "US Government Users Restricted Rights - Use, duplication or "
      + "disclosure restricted by GSA ADP Schedule Contract with " + "IBM Corp.";
  // @COPYRIGHT_END@

  /** The SCCSID which is expanded when the file is extracted from CMVC */
  public static final String sccsid1 = "@(#) com.ibm.mq.jmqi/src/com/ibm/mq/jmqi/samples/JmqiGet.java, jmqi, k701, k701-112-140304 1.7.2.1 09/08/17 08:57:48";

  /**
   * Run the sample JmqiGet program
   * 
   * @param args
   */
  public static void main(String[] args) {

    System.out.println("JmqiGet Start");

    try {
      JmqiGet program = new JmqiGet();
      program.perform(args);
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    System.out.println("JmqiGet End");
  }

  /**
   * Run the program
   * 
   * @param args
   * @throws Exception
   */
  public void perform(String[] args) throws Exception {

    // ****************************************************************
    // * Get the input parameters
    // ****************************************************************
    setOpenOptions(CMQC.MQOO_INPUT_AS_Q_DEF | CMQC.MQOO_FAIL_IF_QUIESCING);
    parseCommandLineArgs(args, 1, 4);
    parseSystemProperties();

    // ****************************************************************
    // * Initialise the Jmqi
    // ****************************************************************
    SampleTraceHandler trace = new SampleTraceHandler();
    JmqiThreadPoolFactory threadPool = new SampleThreadPoolFactory();
    JmqiPropertyHandler propertyHandler = new SamplePropertyHandler();
    JmqiEnvironment env = JmqiFactory.getInstance(trace, threadPool, propertyHandler);

    JmqiMQ mq = getMQInstance(env);

    Pint cc = env.newPint(0);
    Pint rc = env.newPint(0);

    byte[] buffer = new byte[4096];
    ByteBuffer byteBuffer = ByteBuffer.wrap(buffer);
    int bufferLength = buffer.length;
    Pint dataLength = env.newPint(0);

    // ****************************************************************
    // * Connect to queue manager
    // ****************************************************************
    Phconn phconn = env.newPhconn();
    String qmname = getQueueManagerName();
    MQCNO connectOptions = getConnectOptions(env);
    mq.MQCONNX(qmname, connectOptions, phconn, cc, rc);
    if (rc.x != CMQC.MQRC_NONE) {
      throw new Exception("MQCONN ended with reason code " + rc.x);
    }
    Hconn hconn = phconn.getHconn();

    try {
      // ****************************************************************
      // * Open the named message queue for input; exclusive or shared
      // * use of the queue is controlled by the queue definition here
      // ****************************************************************
      MQOD mqod = env.newMQOD();
      mqod.setObjectName(getQueueName());
      int options = getOpenOptions();
      Phobj phobj = env.newPhobj();
      mq.MQOPEN(hconn, mqod, options, phobj, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQOPEN ended with reason code " + rc.x);
      }
      Hobj hobj = phobj.getHobj();

      // ****************************************************************
      // * Get messages from the message queue
      // * Loop until there is a failure
      // ****************************************************************
      MQMD mqmd = env.newMQMD();
      mqmd.setFormat(CMQC.MQFMT_STRING);

      MQGMO mqgmo = env.newMQGMO();
      mqgmo.setVersion(CMQC.MQGMO_VERSION_2);
      mqgmo.setMatchOptions(CMQC.MQMO_NONE);
      mqgmo.setOptions(CMQC.MQGMO_WAIT | CMQC.MQGMO_CONVERT);
      mqgmo.setWaitInterval(15000);

      boolean more = true;
      while (more) {
        mq.MQGET(hconn, hobj, mqmd, mqgmo, bufferLength, byteBuffer, dataLength, cc, rc);
        switch (rc.x) {
          case CMQC.MQRC_NONE :
            byte[] message = byteBuffer.array();
            String messageText = new String(message, 0, dataLength.x);
            System.out.println("message <" + messageText + ">");
            break;
          case CMQC.MQRC_NO_MSG_AVAILABLE :
            System.out.println("no more messages");
            more = false;
            break;
          default :
            throw new Exception("MQGET ended with reason code " + rc.x);
        }
      }

      // ****************************************************************
      // * Close the source queue (if it was opened)
      // ****************************************************************
      options = getCloseOptions();
      mq.MQCLOSE(hconn, phobj, options, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQCLOSE ended with reason code " + rc.x);
      }
    }
    finally {
      // ****************************************************************
      // * Make sure to disconnect from the Queue Manager
      // ****************************************************************
      mq.MQDISC(phconn, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQDISC ended with reason code " + rc.x);
      }
    }
  }
}
