/*
 * <N_OCO_COPYRIGHT>
 * Licensed Materials - Property of IBM
 * 
 * 5724-H72, 5655-R36, 5724-L26, 5655-L82     
 * 
 * (c) Copyright IBM Corp. 2008 All Rights Reserved.
 * 
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with
 * IBM Corp.
 * <NOC_COPYRIGHT>
 */

/**
 *   "JmqiPut" is a sample JMQI program to put messages on a message    
 *   queue, and is an example of the use of MQPUT.                  
 *                                                                  
 *      -- messages are sent to the queue named by the parameter    
 *                                                                  
 *      -- gets lines from StdIn, and adds each to target
 *         queue, taking each line of text as the content           
 *         of a datagram message; the sample stops when a null      
 *         line (or EOF) is read.                                   
 *         New-line characters are removed.                         
 *                                                                  
 *      -- writes a message for each MQI reason other than          
 *         MQRC_NONE; stops if there is a MQI completion code       
 *         of MQCC_FAILED                                           
 *                                                                  
 *    Program logic:                                                
 *         MQOPEN target queue for OUTPUT                           
 *         while end of input file not reached,                     
 *         .  read next line of text                                
 *         .  MQPUT datagram message with text line as data         
 *         MQCLOSE target queue                                     
 *
 *
 *
 * "JmqiPut" has the following parameters                          
 *       required:                                                  
 *                 (1) The name of the target queue                 
 *       optional:                                                  
 *                 (2) Queue manager name                           
 *                 (3) The open options                             
 *                 (4) The close options                            
 *                 (5) The name of the target queue manager         
 *                 (6) The name of the dynamic queue                                          
 *                                                                                                         
 *
 * To connect to the Queue manager in Client mode, specify "MQSERVER" as an environment variable 
 * or a system parameter as follows:
 *     MQSERVER=ChannelName/TransportType/ConnectionName                                                                                                          
 *                                                                                                         
 *                                                      
 *                                                                                                         
 * For example:
 *     bindings mode:
 *         java com.ibm.mq.jmqi.samples.JmqiPut QUEUE QMGR
 *         
 *     client mode:
 *         java -DMQSERVER=SYSTEM.DEF.SVRCONN/TCP/localhost(1414) com.ibm.mq.jmqi.samples.JmqiPut QUEUE QMGR          
 */
package com.ibm.mq.jmqi.samples;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import com.ibm.mq.constants.CMQC;
import com.ibm.mq.jmqi.JmqiEnvironment;
import com.ibm.mq.jmqi.JmqiFactory;
import com.ibm.mq.jmqi.JmqiMQ;
import com.ibm.mq.jmqi.JmqiPropertyHandler;
import com.ibm.mq.jmqi.JmqiThreadPoolFactory;
import com.ibm.mq.jmqi.MQCNO;
import com.ibm.mq.jmqi.MQMD;
import com.ibm.mq.jmqi.MQOD;
import com.ibm.mq.jmqi.MQPMO;
import com.ibm.mq.jmqi.handles.Hconn;
import com.ibm.mq.jmqi.handles.Hobj;
import com.ibm.mq.jmqi.handles.Phconn;
import com.ibm.mq.jmqi.handles.Phobj;
import com.ibm.mq.jmqi.handles.Pint;

/**
 * Sample program to put a message to a queue
 */
public class JmqiPut extends SampleFramework {

  // @COPYRIGHT_START@
  /** Comment for copyright_notice */
  static final String copyright_notice = "Licensed Materials - Property of IBM "
      + "5724-H72, 5655-R36, 5724-L26, 5655-L82                "
      + "(c) Copyright IBM Corp. 2008 All Rights Reserved. "
      + "US Government Users Restricted Rights - Use, duplication or "
      + "disclosure restricted by GSA ADP Schedule Contract with " + "IBM Corp.";
  // @COPYRIGHT_END@

  /** The SCCSID which is expanded when the file is extracted from CMVC */
  public static final String sccsid1 = "@(#) com.ibm.mq.jmqi/src/com/ibm/mq/jmqi/samples/JmqiPut.java, jmqi, k701, k701-112-140304 1.6.2.1 09/08/17 08:58:00";

  /**
   * Run the sample JmqiGet program
   * 
   * @param args
   */
  public static void main(String[] args) {

    System.out.println("JmqiPut Start");

    try {
      JmqiPut program = new JmqiPut();
      program.perform(args);
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    System.out.println("JmqiPut End");
  }

  /**
   * Run the program
   * 
   * @param args
   * @throws Exception
   */
  public void perform(String[] args) throws Exception {

    // ****************************************************************
    // * Get the input parameters
    // ****************************************************************
    setOpenOptions(CMQC.MQOO_OUTPUT | CMQC.MQOO_FAIL_IF_QUIESCING);
    parseCommandLineArgs(args, 1, 6);
    parseSystemProperties();

    // ****************************************************************
    // * Initialise the Jmqi
    // ****************************************************************
    SampleTraceHandler trace = new SampleTraceHandler();
    JmqiThreadPoolFactory threadPool = new SampleThreadPoolFactory();
    JmqiPropertyHandler propertyHandler = new SamplePropertyHandler();
    JmqiEnvironment env = JmqiFactory.getInstance(trace, threadPool, propertyHandler);

    JmqiMQ mq = getMQInstance(env);

    Pint cc = env.newPint(0);
    Pint rc = env.newPint(0);

    // ****************************************************************
    // * Connect to queue manager
    // ****************************************************************
    Phconn phconn = env.newPhconn();
    String qmname = getQueueManagerName();
    MQCNO connectOptions = getConnectOptions(env);
    mq.MQCONNX(qmname, connectOptions, phconn, cc, rc);
    if (rc.x != CMQC.MQRC_NONE) {
      throw new Exception("MQCONN ended with reason code " + rc.x);
    }
    Hconn hconn = phconn.getHconn();

    try {
      // ****************************************************************
      // * Open the target message queue for output
      // ****************************************************************
      MQOD mqod = env.newMQOD();
      mqod.setObjectName(getQueueName());
      mqod.setObjectQMgrName(getTargetQueueManagerName());
      mqod.setDynamicQName(getDynamicQueueName());
      int options = getOpenOptions();
      Phobj phobj = env.newPhobj();
      mq.MQOPEN(hconn, mqod, options, phobj, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQOPEN ended with reason code " + rc.x);
      }
      Hobj hobj = phobj.getHobj();

      // ****************************************************************
      // Read lines from the file and put them to the message queue
      // Loop until null line or end of file, or there is a failure
      // ****************************************************************
      MQMD mqmd = env.newMQMD();
      mqmd.setFormat(CMQC.MQFMT_STRING);

      MQPMO mqpmo = env.newMQPMO();

      InputStreamReader in = new InputStreamReader(System.in);
      BufferedReader stdin = new BufferedReader(in);
      String message;

      while ((message = stdin.readLine()).length() > 0) {

        byte[] buffer = message.getBytes();
        ByteBuffer byteBuffer = ByteBuffer.wrap(buffer);
        int bufferLength = buffer.length;
        mq.MQPUT(hconn, hobj, mqmd, mqpmo, bufferLength, byteBuffer, cc, rc);
        if (rc.x != CMQC.MQRC_NONE) {
          throw new Exception("Could not put the message to the Queue");
        }
      }

      // ****************************************************************
      // * Close the source queue (if it was opened)
      // ****************************************************************
      options = getCloseOptions();
      mq.MQCLOSE(hconn, phobj, options, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQCLOSE ended with reason code " + rc.x);
      }
    }
    finally {
      // ****************************************************************
      // * Make sure to disconnect from the Queue Manager
      // ****************************************************************
      mq.MQDISC(phconn, cc, rc);
      if (rc.x != CMQC.MQRC_NONE) {
        throw new Exception("MQDISC ended with reason code " + rc.x);
      }
    }
  }
}
