/*
 * <N_OCO_COPYRIGHT>
 * Licensed Materials - Property of IBM
 * 
 * 5724-H72, 5655-R36, 5724-L26, 5655-L82     
 * 
 * (c) Copyright IBM Corp. 2008 All Rights Reserved.
 * 
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with
 * IBM Corp.
 * <NOC_COPYRIGHT>
 */
package com.ibm.mq.jmqi.samples;

import com.ibm.mq.constants.CMQC;
import com.ibm.mq.exits.MQCD;
import com.ibm.mq.jmqi.JmqiEnvironment;
import com.ibm.mq.jmqi.JmqiException;
import com.ibm.mq.jmqi.JmqiFactory;
import com.ibm.mq.jmqi.JmqiMQ;
import com.ibm.mq.jmqi.MQCNO;

/**
 * Framework for the sample Jmqi programs
 */
public class SampleFramework {

  // @COPYRIGHT_START@
  /** Comment for copyright_notice */
  static final String copyright_notice = "Licensed Materials - Property of IBM "
      + "5724-H72, 5655-R36, 5724-L26, 5655-L82                "
      + "(c) Copyright IBM Corp. 2008 All Rights Reserved. "
      + "US Government Users Restricted Rights - Use, duplication or "
      + "disclosure restricted by GSA ADP Schedule Contract with " + "IBM Corp.";
  // @COPYRIGHT_END@

  /** The SCCSID which is expanded when the file is extracted from CMVC */
  public static final String sccsid = "@(#) com.ibm.mq.jmqi/src/com/ibm/mq/jmqi/samples/SampleFramework.java, jmqi, k701, k701-112-140304 1.6.2.1 09/08/17 08:58:11";

  private String queueManagerName = "";
  private String queueName = "SYSTEM.DEFAULT.LOCAL.QUEUE";
  private String channelName;
  private String transport;
  private String connectionName;

  private int openOptions;
  private int closeOptions = CMQC.MQCO_NONE;
  private String targetQueueManager = "";
  private String dynamicQueue = "";

  /**
   * Extract the Queue Manager name and the Queue name from the command line arguments
   * 
   * @param args
   * @param min
   * @param max
   * @throws Exception
   */
  public void parseCommandLineArgs(String[] args, int min, int max) throws Exception {

    if (args.length < min) {
      throw new Exception("Not enough arguments");
    }
    else if (args.length > max) {
      throw new Exception("Too many arguments");
    }

    if (args.length >= 1) {
      queueName = args[0];
    }
    if (args.length >= 2) {
      queueManagerName = args[1];
    }
    if (args.length >= 3) {
      openOptions = Integer.parseInt(args[2]);
    }
    if (args.length >= 4) {
      closeOptions = Integer.parseInt(args[3]);
    }
    if (args.length >= 5) {
      targetQueueManager = args[4];
    }
    if (args.length >= 6) {
      dynamicQueue = args[5];
    }
  }

  /**
   * First look for the "MQSERVER" environment variable, then look for the "MQSERVER" environment
   * variable. Then extract the channelName and connectionName from the "MQSERVER" string
   * 
   * The format of the "MQSERVER" string is: MQSERVER=ChannelName/TransportType/ConnectionName
   * 
   * @throws Exception
   */
  public void parseSystemProperties() throws Exception {

    String mqserver = null;

    // First try the "MQSERVER" environment variable.
    // NOTE: "System.getenv()" is not supported on all JMVs.
    // For example, the IBM 1.4.2 JVM throws the following:
    // java.lang.Error: getenv no longer supported, use properties and -D instead: MQSERVER
    try {
      mqserver = System.getenv("MQSERVER");
    }
    catch (Throwable t) {
      // Intentional no-op
      // t.printStackTrace();
    }

    // If we did not find the "MQSERVER" environment variable.
    // then look for the "MQSERVER" system property
    if (mqserver == null) {
      mqserver = System.getProperty("MQSERVER");
    }

    // Extract the channelName and connectionName from the "MQSERVER" string
    if (mqserver != null) {

      String[] words = mqserver.split("/");

      if (words.length == 3) {
        channelName = words[0];
        transport = words[1];
        connectionName = words[2];

        if ("tcp".equals(transport.toLowerCase()) == false) {
          throw new Exception("transport is not valid");
        }
      }
      else {
        throw new Exception("\"MQSERVER\" is not valid");
      }
    }
  }

  /**
   * Return an instance of an JmqiMQ
   * 
   * @param env
   * @return an instance of a JmqiMQ
   * @throws JmqiException
   */
  public JmqiMQ getMQInstance(JmqiEnvironment env) throws JmqiException {

    int bindingType = JmqiFactory.LOCAL_SERVER;
    if ((channelName != null) && (connectionName != null)) {
      bindingType = JmqiFactory.REMOTE;
    }

    int options = 0;
    JmqiMQ mq = env.getMQI(bindingType, options);
    return mq;
  }

  /**
   * @return the name of the Queue Manager
   */
  public String getQueueManagerName() {
    return queueManagerName;
  }

  /**
   * @return the name of the Queue
   */
  public String getQueueName() {
    return queueName;
  }

  /**
   * @param openOptions
   */
  public void setOpenOptions(int openOptions) {
    this.openOptions = openOptions;
  }

  /**
   * @return openOptions
   */
  public int getOpenOptions() {
    return openOptions;
  }

  /**
   * @return closeOptions
   */
  public int getCloseOptions() {
    return closeOptions;
  }

  /**
   * @param closeOptions
   */
  public void setCloseOptions(int closeOptions) {
    this.closeOptions = closeOptions;
  }

  /**
   * @return targetQueueManager
   */
  public String getTargetQueueManagerName() {
    return targetQueueManager;
  }

  /**
   * @return dynamicQueue
   */
  public String getDynamicQueueName() {
    return dynamicQueue;
  }

  /**
   * Compute the MQCNO from the appropriate properties and cache the result in an instance field.
   * 
   * @param env
   * 
   * @return The Connect options to be used when connecting to the Queue manager
   */
  public MQCNO getConnectOptions(JmqiEnvironment env) {
    MQCNO connectOptions = env.newMQCNO();
    if ((connectionName != null) && (channelName != null)) {
      MQCD clientConn = env.newMQCD();
      connectOptions.setClientConn(clientConn);
      clientConn.setConnectionName(connectionName);
      clientConn.setChannelName(channelName);
    }
    return connectOptions;
  }
}
