/*
 * <N_OCO_COPYRIGHT>
 * Licensed Materials - Property of IBM
 * 
 * 5724-H72, 5655-R36, 5724-L26, 5655-L82     
 * 
 * (c) Copyright IBM Corp. 2008 All Rights Reserved.
 * 
 * US Government Users Restricted Rights - Use, duplication or
 * disclosure restricted by GSA ADP Schedule Contract with
 * IBM Corp.
 * <NOC_COPYRIGHT>
 */

package com.ibm.mq.jmqi.samples;

import java.nio.ByteBuffer;

import com.ibm.mq.jmqi.JmqiEnvironment;
import com.ibm.mq.jmqi.JmqiObject;
import com.ibm.mq.jmqi.JmqiTraceHandler;
import com.ibm.mq.jmqi.JmqiTraceHandlerAdapter;
import com.ibm.mq.jmqi.JmqiUtils;

/**
 * Simple Jmqi tracehandler
 */
public class SampleTraceHandler extends JmqiTraceHandlerAdapter {

  // @COPYRIGHT_START@
  /** Comment for copyright_notice */
  static final String copyright_notice = "Licensed Materials - Property of IBM "
      + "5724-H72, 5655-R36, 5724-L26, 5655-L82                "
      + "(c) Copyright IBM Corp. 2008 All Rights Reserved. "
      + "US Government Users Restricted Rights - Use, duplication or "
      + "disclosure restricted by GSA ADP Schedule Contract with " + "IBM Corp.";
  // @COPYRIGHT_END@

  /** The SCCSID which is expanded when the file is extracted from CMVC */
  public static final String sccsid1 = "@(#) com.ibm.mq.jmqi/src/com/ibm/mq/jmqi/samples/SampleTraceHandler.java, jmqi, k701, k701-112-140304 1.6.2.1 09/08/17 08:58:56";

  private static final String tracing = System.getProperty("com.ibm.mq.jmqi.trace");

  /**
   * Constructor
   */
  public SampleTraceHandler() {
    super();
    if ("yes".equalsIgnoreCase(tracing)) {
      isOn = true;
    }
  }

  /**
   * Convert the componentId and functionId to a string representing the class name.
   * 
   * NOTE: Only the Jmqi component (ie COMP_JM and COMP_JO) should ever be converting
   * 
   * @param compID
   * @param funcID
   * @return classname
   */
  private String getClassName(int compID, int funcID) {
    String name = null;
    if (JmqiObject.COMP_JM == compID) {
      name = com.ibm.mq.jmqi.internal.trace.Names.getClassName(funcID);
    }
    else if (JmqiObject.COMP_JO == compID) {
      name = com.ibm.mq.jmqi.internal.trace.MQNames.getClassName(funcID);
    }
    else {
      name = "(" + compID + "." + funcID + ")";
    }
    return name;
  }

  /**
   * Convert the conponentId and functionId to a string representing the method name.
   * 
   * NOTE: Only the Jmqi component (ie COMP_JM and COMP_JO) should ever be converting
   * 
   * @param compID
   * @param funcID
   * @return The method name
   */
  private String getMethodName(int compID, int funcID) {
    String name = null;
    if (JmqiObject.COMP_JM == compID) {
      name = com.ibm.mq.jmqi.internal.trace.Names.getMethodName(funcID);
    }
    else if (JmqiObject.COMP_JO == compID) {
      name = com.ibm.mq.jmqi.internal.trace.MQNames.getMethodName(funcID);
    }
    else {
      name = "(" + compID + "." + funcID + ")";
    }
    return name;
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandlerAdapter#dataFmt(JmqiEnvironment, int, int,
   *      java.lang.String, java.lang.Object)
   */
  public void dataFmt(JmqiEnvironment env, int compID, int funcID, String uniqueDescription,
      Object data) {
    dataFmt(env, null, compID, funcID, uniqueDescription, data);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandlerAdapter#dataFmt(JmqiEnvironment, java.lang.Object, int,
   *      int, java.lang.String, java.lang.Object)
   */
  public void dataFmt(JmqiEnvironment env, Object parentClass, int compID, int funcID,
      String uniqueDescription, Object data) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append(getMethodName(compID, funcID));
      sb.append(": ");

      if (uniqueDescription != null) {
        sb.append(uniqueDescription);
      }

      if (data == null) {
        // empty
      }
      else if (data instanceof ByteBuffer) {
        sb.append(JmqiUtils.NL);
        ByteBuffer byteBuffer = (ByteBuffer) data;
        JmqiUtils.hexDump(null, byteBuffer, 0, byteBuffer.limit(), sb);
      }
      else if (data instanceof byte[]) {
        sb.append(JmqiUtils.NL);
        byte[] byteArray = (byte[]) data;
        JmqiUtils.hexDump(byteArray, null, 0, byteArray.length, sb);
      }
      else {
        sb.append(": ");
        sb.append(data.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param compID
   * @param funcID
   */
  public void entry(int compID, int funcID) {
    entry(null, compID, funcID, null);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   */
  public void entry(Object parentClass, int compID, int funcID) {
    entry(parentClass, compID, funcID, null);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param parameters
   */
  public void entry(Object parentClass, int compID, int funcID, Object[] parameters) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append(" --{ ");
      sb.append(getMethodName(compID, funcID));

      if (parameters != null) {
        for (int i = 0; i < parameters.length; i++) {
          Object parameter = parameters[i];
          sb.append(" ");
          if (parameter == null) {
            sb.append("(null)");
          }
          else {
            sb.append(parameter.toString());
          }
        }
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param compID
   * @param funcID
   * @param returnCode
   */
  public void exit(int compID, int funcID, int returnCode) {
    exit(0, null, compID, funcID, null, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   */
  public void exit(Object parentClass, int compID, int funcID) {
    exit(0, parentClass, compID, funcID, null, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param returnValue
   */
  public void exit(Object parentClass, int compID, int funcID, Object returnValue) {
    exit(0, parentClass, compID, funcID, returnValue, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param exitIndex
   */
  public void exit(Object parentClass, int compID, int funcID, int exitIndex) {
    exit(0, parentClass, compID, funcID, null, exitIndex);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param returnValue
   * @param exitIndex
   */
  public void exit(Object parentClass, int compID, int funcID, Object returnValue, int exitIndex) {
    exit(0, parentClass, compID, funcID, returnValue, exitIndex);
  }

  /**
   * @param compID
   * @param funcID
   * @param uniqueDescription
   * @param data
   */
  public void data(int compID, int funcID, String uniqueDescription, Object data) {
    data(null, compID, funcID, uniqueDescription, data);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param text
   */
  public void trace(Object parentClass, int compID, int funcID, String text) {
    data(parentClass, compID, funcID, text, null);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param text
   * @param data
   */
  public void data(Object parentClass, int compID, int funcID, String text, Object data) {
    data(JmqiTraceHandler.TRACE_USAGE_DATA, parentClass, compID, funcID, text, data);
  }

  /**
   * @param traceLevel
   * @param parentClass
   * @param compID
   * @param funcID
   * @param text
   * @param data
   */
  public void data(int traceLevel, Object parentClass, int compID, int funcID, String text,
      Object data) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("data: ");
      sb.append(getMethodName(compID, funcID));

      if (text != null) {
        sb.append(": ");
        sb.append(text);
      }

      if (data != null) {
        sb.append(": ");
        sb.append(data.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param thrown
   */
  public void catchBlock(Object parentClass, int compID, int funcID, Throwable thrown) {
    catchBlock(parentClass, compID, funcID, thrown, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param thrown
   * @param exitIndex
   */
  public void catchBlock(Object parentClass, int compID, int funcID, Throwable thrown, int exitIndex) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("catch: ");
      sb.append(getMethodName(compID, funcID));
      sb.append("(");
      sb.append(exitIndex);
      sb.append(")");

      if (thrown != null) {
        sb.append(": ");
        sb.append(thrown.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param thrown
   */
  public void throwing(Object parentClass, int compID, int funcID, Throwable thrown) {
    throwing(parentClass, compID, funcID, thrown, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param thrown
   * @param exitIndex
   */
  public void throwing(Object parentClass, int compID, int funcID, Throwable thrown, int exitIndex) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("throwing: ");
      sb.append(getMethodName(compID, funcID));
      sb.append("(");
      sb.append(exitIndex);
      sb.append(")");

      if (thrown != null) {
        sb.append(": ");
        sb.append(thrown.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   */
  public void finallyBlock(Object parentClass, int compID, int funcID) {
    finallyBlock(parentClass, compID, funcID, 0);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param exitIndex
   */
  public void finallyBlock(Object parentClass, int compID, int funcID, int exitIndex) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("finally: ");
      sb.append(getMethodName(compID, funcID));
      sb.append("(");
      sb.append(exitIndex);
      sb.append(")");

      System.out.println(sb.toString());
    }
  }

  /**
   * @param parentClass
   * @param propertyName
   * @param propertyValue
   */
  public void set(Object parentClass, String propertyName, Object propertyValue) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("set ");

      if (propertyName != null) {
        sb.append(": ");
        sb.append(propertyName);
      }

      if (propertyValue != null) {
        sb.append(" = ");
        sb.append(propertyValue.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param parentClass
   * @param propertyName
   * @param propertyValue
   */
  public void get(Object parentClass, String propertyName, Object propertyValue) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append("get ");

      if (propertyName != null) {
        sb.append(": ");
        sb.append(propertyName);
      }

      if (propertyValue != null) {
        sb.append(" = ");
        sb.append(propertyValue.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @param compID
   * @param funcID
   * @param probe
   * @param rc
   * @param insert1
   * @param insert2
   * @param insert3
   * @param insert4
   * @param insert5
   */
  public void ffst(int compID, int funcID, int probe, int rc, int insert1, int insert2,
      String insert3, String insert4, String insert5) {
    ffst(null, compID, funcID, probe, rc, insert1, insert2, insert3, insert4, insert5);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param probe
   * @param rc
   * @param insert1
   * @param insert2
   * @param insert3
   * @param insert4
   * @param insert5
   */
  public void ffst(Object parentClass, int compID, int funcID, int probe, int rc, int insert1,
      int insert2, String insert3, String insert4, String insert5) {
    ffst(parentClass, compID, funcID, probe, rc, insert1, insert2, insert3, insert4, insert5, null);
  }

  /**
   * @param parentClass
   * @param compID
   * @param funcID
   * @param probe
   * @param rc
   * @param insert1
   * @param insert2
   * @param insert3
   * @param insert4
   * @param insert5
   * @param throwable
   */
  public void ffst(Object parentClass, int compID, int funcID, int probe, int rc, int insert1,
      int insert2, String insert3, String insert4, String insert5, Throwable throwable) {
    StringBuffer buffer = new StringBuffer();
    buffer.append("ffst: ");
    buffer.append(getMethodName(compID, funcID));
    buffer.append(", probe: ");
    buffer.append(probe);
    buffer.append(", rc: ");
    buffer.append(rc);
    buffer.append(", insert1: ");
    buffer.append(insert1);
    buffer.append(", insert2: ");
    buffer.append(insert2);

    buffer.append(", insert3: ");
    if (insert3 == null) {
      buffer.append("(null)");
    }
    else {
      buffer.append(insert3);
    }

    buffer.append(", insert4: ");
    if (insert4 == null) {
      buffer.append("(null)");
    }
    else {
      buffer.append(insert4);
    }

    buffer.append(", insert5: ");
    if (insert5 == null) {
      buffer.append("(null)");
    }
    else {
      buffer.append(insert5);
    }
    System.out.println(buffer.toString());

    // Print out a stack trace
    Exception stack = new Exception(throwable);
    stack.fillInStackTrace();
    stack.printStackTrace();
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#entry(int, int, java.lang.Object[])
   */
  public void entry(int compID, int funcID, Object[] parameters) {
    entry(null, compID, funcID, parameters);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int)
   */
  public void exit(int compID, int funcID) {
    exit(0, null, compID, funcID, null, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int, java.lang.Object)
   */
  public void exit(int compID, int funcID, Object returnValue) {
    exit(null, compID, funcID, returnValue, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int, java.lang.Object, int)
   */
  public void exit(int compID, int funcID, Object returnValue, int exitIndex) {
    exit(null, compID, funcID, returnValue, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#catchBlock(int, int, java.lang.Throwable)
   */
  public void catchBlock(int compID, int funcID, Throwable thrown) {
    catchBlock(null, compID, funcID, thrown, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#catchBlock(int, int, java.lang.Throwable, int)
   */
  public void catchBlock(int compID, int funcID, Throwable thrown, int exitIndex) {
    catchBlock(null, compID, funcID, thrown, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#throwing(int, int, java.lang.Throwable)
   */
  public void throwing(int compID, int funcID, Throwable thrown) {
    throwing(null, compID, funcID, thrown, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#throwing(int, int, java.lang.Throwable, int)
   */
  public void throwing(int compID, int funcID, Throwable thrown, int exitIndex) {
    throwing(null, compID, funcID, thrown, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#finallyBlock(int, int)
   */
  public void finallyBlock(int compID, int funcID) {
    finallyBlock(null, compID, funcID, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#finallyBlock(int, int, int)
   */
  public void finallyBlock(int compID, int funcID, int exitIndex) {
    finallyBlock(null, compID, funcID, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#entry_OO(int, int)
   */
  public int entry_OO(int compID, int funcID) {
    entry(null, compID, funcID, null);
    return 0;
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#entry_OO(int, int, java.lang.Object[])
   */
  public int entry_OO(int compID, int funcID, Object[] parameters) {
    entry(null, compID, funcID, parameters);
    return 0;
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#entry_OO(java.lang.Object, int, int)
   */
  public int entry_OO(Object parentClass, int compID, int funcID) {
    entry(parentClass, compID, funcID, null);
    return 0;
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#entry_OO(java.lang.Object, int, int, java.lang.Object[])
   */
  public int entry_OO(Object parentClass, int compID, int funcID, Object[] parameters) {
    entry(parentClass, compID, funcID, parameters);
    return 0;
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int, int, int)
   */
  public void exit(int ctx, int compID, int funcID, int exitIndex) {
    exit(null, compID, funcID, null, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int, int, java.lang.Object)
   */
  public void exit(int ctx, int compID, int funcID, Object returnValue) {
    exit(null, compID, funcID, returnValue, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, int, int, java.lang.Object, int)
   */
  public void exit(int ctx, int compID, int funcID, Object returnValue, int exitIndex) {
    exit(null, compID, funcID, returnValue, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, java.lang.Object, int, int)
   */
  public void exit(int ctx, Object parentClass, int compID, int funcID) {
    exit(null, compID, funcID, null, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, java.lang.Object, int, int, int)
   */
  public void exit(int ctx, Object parentClass, int compID, int funcID, int exitIndex) {
    exit(parentClass, compID, funcID, null, exitIndex);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, java.lang.Object, int, int, java.lang.Object)
   */
  public void exit(int ctx, Object parentClass, int compID, int funcID, Object returnValue) {
    exit(parentClass, compID, funcID, returnValue, 0);
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit(int, java.lang.Object, int, int, java.lang.Object,
   *      int)
   */
  public void exit(int ctx, Object parentClass, int compID, int funcID, Object returnValue,
      int exitIndex) {
    if (isOn) {
      StringBuffer sb = new StringBuffer();
      sb.append(" --} ");
      sb.append(getMethodName(compID, funcID));
      sb.append("(");
      sb.append(ctx);
      sb.append(".");
      sb.append(exitIndex);
      sb.append(")");

      if (returnValue != null) {
        sb.append(" --> ");
        sb.append(returnValue.toString());
      }

      System.out.println(sb.toString());
    }
  }

  /**
   * @see com.ibm.mq.jmqi.JmqiTraceHandler#exit_OO(int, int, int)
   */
  public void exit_OO(int ctx, int compID, int funcID) {
    exit(0, null, compID, funcID, null, 0);
  }
}
